/* $Id: cim_usb.c,v 1.25 2009-01-27 17:06:40 potyra Exp $ 
 *
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "cim_usb.h"

#define USB_MSG_TYPE_POWER              1
#define USB_MSG_TYPE_RESET              2
#define USB_MSG_TYPE_SPEED              3
#define USB_MSG_TYPE_USB_TOKEN          4
#define USB_MSG_TYPE_USB_SOF            5
#define USB_MSG_TYPE_USB_DATA           6
#define USB_MSG_TYPE_USB_HANDSHAKE      7
#define USB_MSG_TYPE_USB_SPECIAL        8

#define USB_RAW_DATA_LENGTH          1023

struct usb_msg {
	int msg_type;

	union {
		struct {
			unsigned int val;
		} power;
		struct {
			int dummy;
		} reset;
		struct {
			int usb_speed;
		} speed;
		struct {
			unsigned char pid;
			unsigned char addr;
			unsigned char endp;
			unsigned char crc5;
		} usb_token;
		struct {
			unsigned short frame_num;
			unsigned char crc5;
		} usb_sof;
		struct {
			unsigned char pid;
			unsigned int length; /* not contained in real packets */
			unsigned char data[USB_RAW_DATA_LENGTH];
			unsigned short crc16;
		} usb_data;
		struct {
			unsigned char pid;
		} usb_handshake;
	} content;
};

static void
cim_usb_power_set(void *_css, unsigned int val)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg msg;
	
	msg.msg_type = USB_MSG_TYPE_POWER;
	msg.content.power.val = val;
	
	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_usb_reset_set(void *_css, /*unsigned*/ int val)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg msg;

	if (val) {
		msg.msg_type = USB_MSG_TYPE_RESET;

		cim_send(&css->bridge, &msg, sizeof(msg));
	}
}

static void
cim_usb_speed_set(void *_css, /*unsigned*/ int val)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg msg;

	msg.msg_type = USB_MSG_TYPE_SPEED;
	msg.content.speed.usb_speed = val;

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_usb_recv_token(
	void *_css,
	int pid,
	int addr,
	int endp
)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg msg;

	assert(! (addr & ~0x7f));
	assert(! (endp & ~0x0f));

	msg.msg_type = USB_MSG_TYPE_USB_TOKEN;
	msg.content.usb_token.pid = pid;
	msg.content.usb_token.addr = addr & 0x7F;
	msg.content.usb_token.endp = endp & 0x0F;
	/* TODO: generate CRC? */

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_usb_recv_sof(
	void *_css,
	int frame_num
)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg msg;

	msg.msg_type = USB_MSG_TYPE_USB_SOF;
	msg.content.usb_sof.frame_num = frame_num;

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_usb_recv_data(
	void *_css,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg msg;

	assert(length <= sizeof(msg.content.usb_data.data));

	msg.msg_type = USB_MSG_TYPE_USB_DATA;
	msg.content.usb_data.pid = pid;
	msg.content.usb_data.length = length;
	memcpy(msg.content.usb_data.data, data, length);

	/* TODO: generate CRC? */

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_usb_recv_handshake(void *_css, int pid)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg msg;

	msg.msg_type = USB_MSG_TYPE_USB_HANDSHAKE;
	msg.content.usb_handshake.pid = pid;

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_usb_interrupt(void *_css, void *_buf, unsigned int bufsize)
{
	struct cim_usb *css = (struct cim_usb *) _css;
	struct usb_msg *msg = (struct usb_msg *) _buf;

	assert(bufsize == sizeof(*msg));

	switch (msg->msg_type) {
	case USB_MSG_TYPE_POWER:
		sig_boolean_set(css->sig_power, css,
				msg->content.power.val);
		break;

	case USB_MSG_TYPE_RESET:
		sig_usb_bus_reset_set(css->sig_bus, css, 1);
		sig_usb_bus_reset_set(css->sig_bus, css, 0);
		break;

	case USB_MSG_TYPE_SPEED:
		sig_usb_bus_speed_set(css->sig_bus, css,
				msg->content.speed.usb_speed);
		break;

	case USB_MSG_TYPE_USB_TOKEN:
		sig_usb_bus_send_token(css->sig_bus, css,
				msg->content.usb_token.pid,
				msg->content.usb_token.addr,
				msg->content.usb_token.endp);
		break;

	case USB_MSG_TYPE_USB_SOF:
		assert(0);

	case USB_MSG_TYPE_USB_SPECIAL:
		assert(0);

	case USB_MSG_TYPE_USB_DATA:
		sig_usb_bus_send_data(css->sig_bus, css,
				msg->content.usb_data.pid,
				msg->content.usb_data.length,
				msg->content.usb_data.data,
				msg->content.usb_data.crc16);
		break;

	case USB_MSG_TYPE_USB_HANDSHAKE:
		sig_usb_bus_send_handshake(css->sig_bus, css,
				msg->content.usb_handshake.pid);
		break;
	default:
		assert(0);
	}
}

void
cim_usb_init(
	struct cim_usb *css,
	struct sig_usb_bus *usb_bus
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = cim_usb_power_set,
	};
	static const struct sig_usb_bus_main_funcs bus_funcs = {
		.reset_set = cim_usb_reset_set,
		.speed_set = cim_usb_speed_set,
		.recv_token = cim_usb_recv_token,
		.recv_sof = cim_usb_recv_sof,
		.recv_data = cim_usb_recv_data,
		.recv_handshake = cim_usb_recv_handshake,
	};

	/* Call */
	css->sig_bus = usb_bus->bus;
	sig_usb_bus_main_connect(usb_bus->bus, css, &bus_funcs);

	/* Out */
	css->sig_power = usb_bus->power;
	sig_boolean_connect_out(usb_bus->power, css, 0);

	/* In */
	sig_boolean_connect_in(usb_bus->power, css, &power_funcs);

	cim_connect(&css->bridge, cim_usb_interrupt, css);
}

void
cim_usb_create(struct cim_usb *css)
{
	cim_create(&css->bridge);
}

void
cim_usb_destroy(struct cim_usb *css)
{
	cim_destroy(&css->bridge);
}
