/*
 * $Id: harddisk.c,v 1.1 2012-07-03 07:51:36 vrsieh Exp $ 
 *
 * Copyright (C) 2004-2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

/* ===================== RUNTIME || INIT ======================== */
#if defined(RUNTIME_RM) || defined(INIT_RM)

#define IDE_ERR_STAT	0x01
#define IDE_INDEX_STAT	0x02
#define IDE_ECC_STAT	0x04
#define IDE_DRQ_STAT	0x08
#define IDE_SEEK_STAT	0x10
#define IDE_WRERR_STAT	0x20
#define IDE_READY_STAT	0x40
#define IDE_BUSY_STAT	0x80

#define WIN_RESTORE	0x10 /* Recalibrate Device */

extern const uint16_t ide_port_table[];

#endif /* RUNTIME_RM || INIT_RM */
/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

CODE16;

#include "assert.h"
#include "fixme.h"
#include "stdio.h"
#include "string.h"
#include "in.h"

#include "io.h"
#include "const.h"
#include "cmos.h"
#include "var.h"
#include "harddisk.h"

uint8_t
ide_harddisk_read_chs(
	uint8_t device,
	uint8_t num, 
	uint16_t cylinder, 
	uint8_t head,
	uint8_t sector,
	uint16_t buffer_seg,
	uint16_t buffer_off)
{
	uint16_t port;
	uint8_t status;

	assert(/* 0 <= device / 2 && */ device / 2 < 6);
	assert(/* 0 <= device % 2 && */ device % 2 < 2);

	/* Get port addresses. */
	port = const_get(ide_port_table[device / 2]);

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	/*
	 * Send read command.
	 */
	outb(num, port + 2);		/* number of sectors */
	outb(sector & 0x3f, port + 3);	/* sector */
	outb(cylinder & 0xff, port + 4);/* cylinder (low bits) */
	outb(cylinder >> 8, port + 5);	/* cylinder (high bits) */
	outb(0xa0 | ((device % 2) << 4) | head, port + 6);
					/* unit / head */
	outb(0x20, port + 7);		/* command: read */

	/*
	 * Get all sectors.
	 */
	for ( ; 0 < num; num--) {
		uint16_t count;
		uint16_t data;

		/* Wait while busy. */
		do {
			status = inb(port + 0x206);
		} while (status & IDE_BUSY_STAT);

		if (status & IDE_ERR_STAT) {
			/* ERR bit set */
			assert(0);
			return 0xff;	/* FIXME VOSSI */
		}

		if (! (status & IDE_DRQ_STAT)) {
			/* FIXME VOSSI */
			assert(0);
			return 0xff;
		}

		buffer_seg += buffer_off / 16;
		buffer_off %= 16;

		for (count = 0; count < 512 / 2; count++) {
			data = inw(port);
			put_word(buffer_seg, buffer_off + count * 2, data);
		}

		buffer_seg += 512 / 16;
	}

	return 0;
}

uint8_t
ide_harddisk_write_chs(
	uint8_t device,
	uint8_t num, 
	uint16_t cylinder, 
	uint8_t head,
	uint8_t sector,
	uint16_t buffer_seg,
	uint16_t buffer_off
)
{
	uint16_t port;
	uint8_t status;

	assert(/* 0 <= device / 2 && */ device / 2 < 6);
	assert(/* 0 <= device % 2 && */ device % 2 < 2);

	/* Get port addresses. */
	port = const_get(ide_port_table[device / 2]);

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	/*
	 * Send write command.
	 */
	outb(num, port + 2);		/* number of sectors */
	outb(sector & 0x3f, port + 3);	/* sector */
	outb(cylinder & 0xff, port + 4);/* cylinder (low bits) */
	outb(cylinder >> 8, port + 5);	/* cylinder (high bits) */
	outb(0xa0 | ((device % 2) << 4) | head, port + 6);
					/* unit / head */
	outb(0x30, port + 7);		/* command: write */

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	/*
	 * Send all sectors.
	 */
	for ( ; 0 < num; num--) {
		uint16_t count;
		uint16_t data;

		if (! (status & IDE_DRQ_STAT)) {
			/* FIXME VOSSI */
			assert(0);
			return 0xff;
		}

		/* Send sector. */
		buffer_seg += buffer_off / 16;
		buffer_off %= 16;

		for (count = 0; count < 512 / 2; count++) {
			data = get_word(buffer_seg, buffer_off + count * 2);
			outw(data, port);
		}

		buffer_seg += 512 / 16;

		/* Wait while busy. */
		do {
			status = inb(port + 0x206);
		} while (status & IDE_BUSY_STAT);

		if (status & IDE_ERR_STAT) {
			/* ERR bit set */
			assert(0);
			return 0xff;	/* FIXME VOSSI */
		}
	}

	return 0;
}

uint8_t
ide_harddisk_read_lba(
	uint8_t controller,
	uint8_t unit,
	uint8_t num, 
	uint32_t /* long long */ start,
	uint16_t buffer_seg,
	uint16_t buffer_off
) {
	uint16_t port;
	uint8_t status;

	assert(/* 0 <= controller && */ controller < 6);
	assert(/* 0 <= unit && */ unit < 2);

	/* Get port addresses. */
	port = const_get(ide_port_table[controller]);

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	/*
	 * Send read command.
	 */
	outb(num, port + 2);			/* number of sectors */
	outb((start >> 0) & 0xff, port + 3);	/* sector */
	outb((start >> 8) & 0xff, port + 4);	/* cylinder (low bits) */
	outb((start >> 16) & 0xff, port + 5);	/* cylinder (high bits) */
	outb(0xe0 | (unit << 4) | ((start >> 24) & 0xf), port + 6);
						/* unit / head */
	outb(0x20, port + 7);			/* command: read */

	/*
	 * Get all sectors.
	 */
	for ( ; 0 < num; num--) {
		uint16_t count;
		uint16_t data;

		/* Wait while busy. */
		do {
			status = inb(port + 0x206);
		} while (status & IDE_BUSY_STAT);

		if (status & IDE_ERR_STAT) {
			/* ERR bit set */
			return 0xff;    /* FIXME VOSSI */
		}

		/* Read result. */
		if (! (status & IDE_DRQ_STAT)) {
			/* FIXME VOSSI */
			assert(0);
			return 0xff;
		}

		buffer_seg += buffer_off / 16;
		buffer_off %= 16;

		for (count = 0; count < 512 / 2; count++) {
			data = inw(port);
			put_word(buffer_seg, buffer_off + count * 2, data);
		}

		buffer_seg += 512 / 16;
	}

	return 0;
}

uint8_t
ide_harddisk_write_lba(
	uint8_t controller,
	uint8_t unit,
	uint8_t num, 
	uint32_t /* long long */ start,
	uint16_t buffer_seg,
	uint16_t buffer_off)
{
	uint16_t port;
	uint8_t status;

	assert(/* 0 <= controller && */ controller < 6);
	assert(/* 0 <= unit && */ unit < 2);

	/* Get port addresses. */
	port = const_get(ide_port_table[controller]);

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	/*
	 * Send write command.
	 */
	outb(num, port + 2);			/* number of sectors */
	outb((start >> 0) & 0xff, port + 3);	/* sector */
	outb((start >> 8) & 0xff, port + 4);	/* cylinder (low bits) */
	outb((start >> 16) & 0xff, port + 5);	/* cylinder (high bits) */
	outb(0xe0 | (unit << 4) | ((start >> 24) & 0xf), port + 6);
						/* unit / head */
	outb(0x30, port + 7);			/* command: write */

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	/*
	 * Send all sectors.
	 */
	for ( ; 0 < num; num--) {
		uint16_t count;
		uint16_t data;

		if (! (status & IDE_DRQ_STAT)) {
			/* FIXME VOSSI */
			assert(0);
			return 0xff;
		}

		/* Write sector. */
		buffer_seg += buffer_off / 16;
		buffer_off %= 16;

		for (count = 0; count < 512 / 2; count++) {
			data = get_word(buffer_seg, buffer_off + count * 2);
			outw(data, port);
		}

		buffer_seg += 512 / 16;

		/* Wait while busy. */
		do {
			status = inb(port + 0x206);
		} while (status & IDE_BUSY_STAT);

		if (status & IDE_ERR_STAT) {
			/* ERR bit set */
			return 0xff;    /* FIXME VOSSI */
		}
	}

	return 0;
}

#endif /* RUNTIME_RM */
/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "assert.h"
#include "stdio.h"
#include "string.h"
#include "in.h"
#include "io.h"
#include "cmos.h"
#include "var.h"
#include "const.h"
#include "harddisk.h"

uint8_t
recalibrate(uint8_t contr, uint8_t drive)
{
	uint16_t port;
	uint8_t status;

	assert(/* 0 <= contr && */ contr < 6);
	assert(/* 0 <= drive && */ drive < 2);

	port = const_get(ide_port_table[contr]);

	/* Send command. */
	outb(0, port + 1);		/* Features */
	outb(1, port + 2);		/* #Sectors */
	outb(0, port + 3);		/* Sector Number */
	outb(0, port + 4);		/* Cylinder Low */
	outb(0, port + 5);		/* Cylinder High */
	outb(drive << 4, port + 6);	/* Unit */
	outb(WIN_RESTORE, port + 7);	/* Command */

	/* Wait while controller busy. */
	do {
		status = inb(port + 0x206);
		if (status == 0xff) {
			/* No drive present. */
			return 1;
		}
	} while (status & IDE_BUSY_STAT);

	return 0;
}

#endif /* INIT_RM */
