import type { StreamParser } from "@codemirror/language";

// Import the autogenerated (by a script in the 'contrib' directory) lists
import grammar from "./bql-grammar";

const keywords = new Set(grammar.keywords);
const columns = new Set(grammar.columns);
const functions = new Set(grammar.functions);

// This should match the grammar defined in Beancount (`query/query_parser.py`).
const string = /^("[^"]*"|'[^']*')/;
const date = /^(?:#(?:"[^"]*"|'[^']*')|\d\d\d\d-\d\d-\d\d)/;
const decimal = /^[-+]?([0-9]+\.[0-9]*|[0-9]*\.[0-9]+)/;
const integer = /^[-+]?[0-9]+/;

export const bqlStreamParser: StreamParser<unknown> = {
  token(stream) {
    if (stream.eatSpace() || stream.eol()) {
      return null;
    }
    if (stream.match(string)) {
      return "string";
    }
    if (stream.match(date) || stream.match(decimal) || stream.match(integer)) {
      return "number";
    }
    if (stream.match(/\w+/)) {
      const word = stream.current().toLowerCase();
      if (keywords.has(word)) {
        return "keyword";
      }
      if (columns.has(word)) {
        return "typeName";
      }
      if (functions.has(word) && stream.peek() === "(") {
        return "macroName";
      }
      return "name";
    }
    // Skip one character since no known token matched.
    const char = stream.next();
    if (char === "*") {
      return "typeName";
    }
    return null;
  },
};
