from gettext import gettext as _
from gi.repository import Gdk, GObject, Gio, GLib, Gtk

import logging
from typing import List, Optional

from iotas.category import Category, CategorySpecialPurpose
from iotas.category_manager import CategoryManager
import iotas.config_manager
import iotas.const
from iotas.note import Note
from iotas.note_manager import NoteManager
from iotas.selection_header_bar import SelectionHeaderBar
from iotas.sync_manager import SyncManager
from iotas.theme_selector import ThemeSelector


@Gtk.Template(resource_path="/org/gnome/gitlab/cheywood/Iotas/ui/index.ui")
class Index(Gtk.Box):
    __gtype_name__ = "Index"

    __gsignals__ = {
        "note-opened": (GObject.SignalFlags.RUN_FIRST, None, (Note,)),
    }

    UNDO_NOTE_DELETION_DURATION = 5.0
    SYNC_CONFLICT_NOTIFICATION_DURATION = 5.0

    _categories_flap = Gtk.Template.Child()

    _stack = Gtk.Template.Child()
    _sidebar = Gtk.Template.Child()
    _note_list_scroll = Gtk.Template.Child()
    _note_list = Gtk.Template.Child()
    _first_start = Gtk.Template.Child()
    _empty = Gtk.Template.Child()
    _search_info = Gtk.Template.Child()
    _search_empty = Gtk.Template.Child()

    _header_stack = Gtk.Template.Child()
    _main_header_bar = Gtk.Template.Child()
    _sidebar_button = Gtk.Template.Child()
    _search_header_bar = Gtk.Template.Child()
    _selection_header_bar = Gtk.Template.Child()
    _title = Gtk.Template.Child()

    _notification = Gtk.Template.Child()
    _timed_notification = Gtk.Template.Child()

    _menu_button = Gtk.Template.Child()
    _new_button = Gtk.Template.Child()

    _offline_banner = Gtk.Template.Child()

    def __init__(self):
        super().__init__()

        self.__note_manager = None
        self.__category_manager = None
        self.__sync_manager = None

        self.__add_fav_action = None
        self.__remove_fav_action = None
        self.__initialised = False
        self.__create_note_after_init = False
        self.__searching_before_selection = False
        self.__active = False

        # Startup bootstrap category
        self.__category = Category("", 0)
        self.__category.special_purpose = CategorySpecialPurpose.ALL
        self._title.set_label(Category.ALL_TITLE)

        self._note_list.connect("note-opened", self.__on_note_opened)
        self._note_list.connect("note-checkbox-activated", self.__on_note_checkbox_activated)
        self._note_list.connect("note-checkbox-deactivated", self.__on_note_checkbox_deactivated)

        if not iotas.config_manager.get_first_start():
            # Translators: Description, notification
            self._notification.show(_("Loading"), immediate=True)

        self._selection_header_bar.connect("categories-changed", self.__on_notes_categories_changed)
        self._selection_header_bar.connect("delete", self.__on_notes_deleted)
        self._selection_header_bar.connect("set-favourite", self.__on_notes_set_favourite)
        self._selection_header_bar.connect("clear-favourite", self.__on_notes_clear_favourite)
        self._selection_header_bar.connect("abort", self.__on_category_header_bar_abort)

        self._menu_button.get_popover().add_child(ThemeSelector(), "theme")

        self.connect("realize", self.__on_realize)

        controller = Gtk.EventControllerKey()
        controller.connect("key-pressed", self.__on_index_key_pressed)
        self.add_controller(controller)

        self._search_header_bar.connect("changed", self.__on_search_changed)

        iotas.config_manager.settings.connect(
            "changed::index-category-style", self.__category_style_changed
        )

    def setup(
        self,
        note_manager: NoteManager,
        category_manager: CategoryManager,
        sync_manager: SyncManager,
    ) -> None:
        """Perform initial setup.

        :param NoteManager note_manager: Note manager
        :param CategoryManager category_manager: Category manager
        :param SyncManager sync_manager: Remote sync manager
        """
        self.__sync_manager = sync_manager

        self.__note_manager = note_manager
        self.__note_manager.connect(
            "initial-load-complete", self.__on_initial_load_from_db_complete
        )
        self.__note_manager.connect("new-note-persisted", self.__on_new_note_persisted)

        self.__category_manager = category_manager

        self.__setup_actions()

        self.__sync_manager.set_managers(self.__note_manager, self.__category_manager)

        self._sidebar.setup(
            self.__sync_manager,
            self.__category_manager,
            self.__on_category_activated,
        )
        self._selection_header_bar.setup(self.__category_manager.tree_store)
        self._note_list.setup(self.__note_manager, self.__on_listbox_key_pressed)
        self._selection_header_bar.bind_property(
            "active", self._note_list, "selecting", GObject.BindingFlags.SYNC_CREATE
        )
        self._search_header_bar.bind_property(
            "active", self._note_list, "searching", GObject.BindingFlags.SYNC_CREATE
        )

        if iotas.config_manager.get_first_start():
            self._stack.set_visible_child(self._first_start)
        else:
            self._stack.set_visible(False)

        self.__sync_manager.connect("ready", self.__on_sync_ready)
        self.__sync_manager.connect("started", self.__on_sync_started)
        self.__sync_manager.connect("finished", self.__on_sync_finished)

        self.__sync_manager.init_auth()

        self.__sync_manager.bind_property(
            "offline", self._offline_banner, "revealed", GObject.BindingFlags.SYNC_CREATE
        )

    def update_for_note_deletions(self, notes: List[Note]) -> None:
        if len(notes) > 1:
            # Translators: Description, notification, {} is a positive number
            msg = _("{} notes deleted").format(len(notes))
        else:
            # Translators: Description, notification
            msg = _("Note deleted")
        self._timed_notification.show(
            msg,
            self.UNDO_NOTE_DELETION_DURATION,
            # Translators: Button
            _("Undo"),
            self.__undo_deletion,
            self.__note_manager.on_deletion_undo_elapsed,
        )
        if self.__searching():
            self.__apply_search()
        else:
            self.__invalidate_view()
            self.__handle_emptied_category()

    def show_note_conflict_alert(self) -> None:
        """Let the user know a sync conflict has occurred with the note being edited."""
        self._timed_notification.show(
            # Translators: Description, notification
            _("Sync conflict with note being edited"),
            self.SYNC_CONFLICT_NOTIFICATION_DURATION,
        )

    def show_secret_service_failure_alert(self) -> None:
        # Somewhat a clunky placeholder approach for now
        self._timed_notification.show(
            _(
                # Translators: Description, notification. "Secret Service" and "gnome-keyring"
                # should likely not be translated.
                "Failure accessing Secret Service. Ensure you have a "
                "provider like gnome-keyring which has a default keyring "
                "setup that is unlocked."
            ),
            0.0,
            # Translators: Button
            _("OK"),
        )

    def refresh_after_note_closed(self, note: Note) -> None:
        if self.__searching():
            self.__apply_search()
        else:
            self.__invalidate_view()
            if not self.__handle_emptied_category(note):
                self._note_list.update_category_labels(self.__category)

        if self.get_root().using_keyboard_navigation:
            if not note.locally_deleted:
                self._note_list.refocus_selected_row()
        else:
            self._note_list.clear_selections()

    @GObject.Property(type=bool, default=False)
    def active(self) -> bool:
        return self.__active

    @active.setter
    def active(self, value: bool) -> None:
        self.__active = value
        self.__enable_actions(value)

    def __on_search_changed(self, _object: GObject.Object) -> None:
        self.__apply_search()

    def __setup_actions(self) -> None:
        action_group = Gio.SimpleActionGroup.new()
        app = Gio.Application.get_default()

        action = Gio.SimpleAction.new("toggle-sidebar")
        action.connect("activate", self.__on_toggle_sidebar)
        action_group.add_action(action)
        app.set_accels_for_action("index.toggle-sidebar", ["F9"])

        action = Gio.SimpleAction.new("category-context-dependent")
        action.connect("activate", self.__on_category_shortcut)
        action_group.add_action(action)
        app.set_accels_for_action("index.category-context-dependent", ["<Control>e"])

        action = Gio.SimpleAction.new("create-note")
        action.connect("activate", self.__on_create_note)
        action_group.add_action(action)
        app.set_accels_for_action("index.create-note", ["<Control>n"])

        action = Gio.SimpleAction.new("delete-from-index")
        action.connect("activate", self.__on_delete_note_from_keyboard)
        action_group.add_action(action)
        app.set_accels_for_action("index.delete-from-index", ["<Shift>Delete"])

        action = Gio.SimpleAction.new("load-older-notes")
        action.connect("activate", self.__load_older_notes)
        action_group.add_action(action)

        action = Gio.SimpleAction.new("search")
        action.connect("activate", self.__on_search)
        action_group.add_action(action)
        app.set_accels_for_action("index.search", ["<Control>f"])

        action = Gio.SimpleAction.new("select-notes")
        action.connect("activate", self.__on_enter_selection)
        action_group.add_action(action)
        app.set_accels_for_action("index.select-notes", ["<Control>s"])

        action = Gio.SimpleAction.new("cancel")
        action.connect("activate", self.__cancel)
        action_group.add_action(action)
        app.set_accels_for_action("index.cancel", ["Escape", "<Alt>Left"])

        action = Gio.SimpleAction.new("selection-toggle-favourite")
        action.connect("activate", self.__on_toggle_favourite)
        action_group.add_action(action)
        app.set_accels_for_action("index.selection-toggle-favourite", ["<Control>g"])

        self.__action_group = action_group
        app.get_active_window().insert_action_group("index", action_group)

    def __enable_actions(self, enabled: bool) -> None:
        """Toggle whether index actions are enabled.

        :param bool enabled: New value
        """
        actions = self.__action_group.list_actions()
        for action in actions:
            self.__action_group.lookup_action(action).set_enabled(enabled)

    def __load_older_notes(self, _action: Gio.SimpleAction, _param: GLib.Variant) -> None:
        self.__load_older_notes_and_notify()
        self.__invalidate_view()
        self._note_list.update_category_labels(self.__category)

    def __on_realize(self, _widget: Gtk.Widget) -> None:
        self.__note_manager.update_filters(self.__category)
        self.__note_manager.initiate_model_from_db()
        self.__category_manager.populate()

    def __on_create_note(
        self,
        _obj: GObject.Object,
        _value: GObject.ParamSpec,
    ) -> None:
        """Create a new note and edit it."""
        if not self.__initialised:
            self.__create_note_after_init = True
            return
        self.__create_and_open_note()

    def __on_notes_categories_changed(self, _obj: SelectionHeaderBar) -> None:
        changeset = self._selection_header_bar.get_categories_changeset()
        self.__cancel_selection()
        for note, old_category in changeset:
            self.__note_manager.persist_note_category(note, old_category)
        if self.__searching():
            self.__apply_search()
        else:
            self.__invalidate_view()
            self.__handle_emptied_category()

    def __on_notes_clear_favourite(self, _obj: SelectionHeaderBar) -> None:
        notes = self._selection_header_bar.get_selected()
        self.__update_favourites(notes, False)
        self.__cancel_selection()

    def __on_notes_set_favourite(self, _obj: SelectionHeaderBar) -> None:
        notes = self._selection_header_bar.get_selected()
        self.__update_favourites(notes, True)
        self.__cancel_selection()

    def __on_notes_deleted(self, _obj: SelectionHeaderBar) -> None:
        notes = self._selection_header_bar.get_selected()
        self.__delete_notes(notes)
        self.__cancel_selection()

    def __on_note_opened(self, _obj: GObject.Object, note: Note) -> None:
        self.emit("note-opened", note)

    def __on_note_checkbox_activated(self, _obj: GObject.Object, note: Note) -> None:
        if not self._selection_header_bar.active:
            self.__start_selection()
        self._selection_header_bar.set_note_selected(note, True)

    def __on_note_checkbox_deactivated(self, _obj: GObject.Object, note: Note) -> None:
        self._selection_header_bar.set_note_selected(note, False)

    def __on_category_header_bar_abort(self, _obj: SelectionHeaderBar) -> None:
        self.__cancel_selection()

    def __on_category_activated(self, category: Category, close_flap: bool = True) -> None:
        self.__category = category
        self._title.set_label(category.display_name)
        self.__invalidate_view()
        if self.__category.special_purpose is None and not self._note_list.older_notes_loaded:
            self.__load_older_notes_and_notify(
                displayed=False, time_filtered=True, update_section_visibility=True
            )
        else:
            self._note_list.refresh_section_visibility(self.__category)
        self._note_list.update_category_labels(self.__category)
        if close_flap and self._categories_flap.get_reveal_flap():
            self._categories_flap.set_reveal_flap(False)
        if self.get_root().using_keyboard_navigation:
            self._note_list.move_focus_to_list_top()

    def __on_search(self, _action: Gio.SimpleAction, _param: GLib.Variant) -> None:
        self.__enter_search()

    def __on_enter_selection(self, _action: Gio.SimpleAction, _param: GLib.Variant) -> None:
        self.__start_selection()

    def __on_sync_ready(self, _obj: GObject.Object, new_setup: bool) -> None:
        if new_setup and self.__searching():
            self.__end_search()

    def __on_sync_started(self, _obj: GObject.Object) -> None:
        if self.active:
            # Translators: Description, notification
            self._notification.show(_("Syncing"))

    def __on_sync_finished(self, _obj: SyncManager, changes: bool) -> None:
        if self.active:
            self._notification.hide()
            if changes:
                self.__invalidate_view()
                self._note_list.update_category_labels(self.__category)

    def __on_index_key_pressed(
        self,
        controller: Gtk.EventControllerKey,
        keyval: int,
        keycode: int,
        state: Gdk.ModifierType,
    ):
        if keyval in (Gdk.KEY_Down, Gdk.KEY_KP_Down):
            if self._categories_flap.get_reveal_flap():
                self.get_root().using_keyboard_navigation = True
                self._sidebar.take_focus()
            else:
                self._note_list.move_focus_to_list_top()
            return Gdk.EVENT_STOP
        elif keyval in (Gdk.KEY_Up, Gdk.KEY_KP_Up):
            self._note_list.move_focus_to_list_top()
            return Gdk.EVENT_STOP
        elif not self.__searching() and not self._selection_header_bar.active:
            result = self._search_header_bar.check_if_starting(controller, keyval, state)
            if result == Gdk.EVENT_STOP:
                self.__enter_search(False)
            return result

        return Gdk.EVENT_PROPAGATE

    def __on_listbox_key_pressed(
        self,
        controller: Gtk.EventControllerKey,
        keyval: int,
        keycode: int,
        state: Gdk.ModifierType,
    ):
        if keyval in (Gdk.KEY_Down, Gdk.KEY_KP_Down):
            self._note_list.focus_next_list_row(self.__get_focused())
            return Gdk.EVENT_STOP
        elif keyval in (Gdk.KEY_Up, Gdk.KEY_KP_Up):
            self._note_list.focus_previous_list_row(self.__get_focused())
            return Gdk.EVENT_STOP
        elif not self.__searching() and not self._selection_header_bar.active:
            result = self._search_header_bar.check_if_starting(controller, keyval, state)
            if result == Gdk.EVENT_STOP:
                self.__enter_search(False)
            return result

        return Gdk.EVENT_PROPAGATE

    def __on_delete_note_from_keyboard(
        self, _action: Gio.SimpleAction, _param: GLib.Variant
    ) -> None:
        if self._selection_header_bar.active:
            self._selection_header_bar.handle_delete_keyboard_shortcut()
        else:
            note = None
            window = self.get_root()
            focused = window.get_focus()
            if isinstance(focused, Gtk.ListBoxRow):
                note = focused.get_first_child().note
            if note:
                self.__delete_notes([note])

    def __on_toggle_favourite(self, _action: Gio.SimpleAction, _param: GLib.Variant) -> None:
        if self._selection_header_bar.active:
            self._selection_header_bar.toggle_favourite_on_selection()

    def __on_category_shortcut(self, _action: Gio.SimpleAction, _param: GLib.Variant) -> None:
        if self._selection_header_bar.active:
            self._selection_header_bar.edit_category_for_selection()
        else:
            self.__toggle_sidebar()

    def __on_toggle_sidebar(self, _action: Gio.SimpleAction, _param: GLib.Variant) -> None:
        self.__toggle_sidebar()

    def __create_and_open_note(self) -> None:
        note = self.__note_manager.create_note(self.__category)
        self.emit("note-opened", note)

    def __invalidate_view(self) -> None:
        """Refresh the view."""
        self.__note_manager.invalidate_sort()
        self.__note_manager.update_filters(self.__category)

        # Update display of empty state, list, etc
        if self.__searching():
            if self._note_list.older_notes_loaded_but_empty:
                self._stack.set_visible_child(self._search_empty)
        else:
            include_older = self._note_list.older_notes_displayed or self.__is_showing_category()
            count = self.__note_manager.get_filtered_note_count(include_older)
            if not iotas.config_manager.get_first_start():
                if count == 0:
                    self._stack.set_visible_child(self._empty)
                elif self._stack.get_visible_child() in (
                    self._empty,
                    self._first_start,
                ):
                    self._stack.set_visible_child(self._note_list_scroll)

    def __enter_search(self, clear_text: bool = True) -> None:
        """Enter the search UI."""
        if self.__searching():
            return

        self._search_header_bar.enter(clear_text)

        all_category = self._sidebar.select_and_fetch_category(None, CategorySpecialPurpose.ALL)
        self.__category = all_category
        if self._categories_flap.get_reveal_flap():
            self._categories_flap.set_reveal_flap(False)

        self.__set_search_visible(True)
        if not self._note_list.older_notes_loaded:
            self.__load_older_notes_and_notify(displayed=False, time_filtered=False)

    def __apply_search(self) -> None:
        if not self.__searching():
            return
        # If trying to search without caching completed delay the search
        if not self._note_list.older_notes_loaded:
            GLib.timeout_add(500, self.__apply_search)
            logging.debug("Delaying search due to model not ready")
            return
        search_text = self._search_header_bar.text
        if len(search_text) > 1:
            found_ids = self.__note_manager.search_notes(search_text)
            if len(found_ids) > 0:
                self._note_list.restrict_for_search_by_ids(found_ids)
                self.__invalidate_view()
                self._note_list.update_category_labels(self.__category)
                self._stack.set_visible_child(self._note_list_scroll)
                self._note_list_scroll.get_vadjustment().set_value(0)
            else:
                self._stack.set_visible_child(self._search_empty)
        else:
            self._stack.set_visible_child(self._search_info)
            self._note_list.restrict_for_search_by_ids(None)

    def __start_selection(self) -> None:
        if self._categories_flap.get_reveal_flap():
            self._categories_flap.set_reveal_flap(False)
        self.__searching_before_selection = self.__searching()
        self._header_stack.set_visible_child(self._selection_header_bar)
        # Prevent focus jumping to top row of listbox and resulting scroll
        if not self.get_root().using_keyboard_navigation:
            self._sidebar_button.grab_focus()
        self._selection_header_bar.activate()
        self.__update_enabled_actions(False)

    def __cancel_selection(self) -> None:
        if self.__searching_before_selection:
            self._header_stack.set_visible_child(self._search_header_bar)
        else:
            self._header_stack.set_visible_child(self._main_header_bar)
        self._selection_header_bar.active = False
        self._note_list.clear_all_checkboxes()
        self.__update_enabled_actions(True)

        if self.get_root().using_keyboard_navigation:
            self._note_list.refocus_selected_row()
        else:
            self._note_list.clear_selections()
            # Prevent focus jumping to top row of listbox and resulting scroll
            self._sidebar_button.grab_focus()

    def __searching(self) -> bool:
        return self._search_header_bar.active

    def __set_search_visible(self, visible: bool) -> None:
        if visible:
            self._header_stack.set_visible_child(self._search_header_bar)
        else:
            self._header_stack.set_visible_child(self._main_header_bar)
        self._note_list.refresh_section_visibility(self.__category)
        self._note_list.update_older_notes_loader_visibility_for_search(visible)
        if visible:
            self._stack.set_visible_child(self._search_info)
        else:
            self._stack.set_visible_child(self._note_list_scroll)
            if self._note_list.older_notes_loaded:
                self._note_list.restrict_for_search_by_ids(None)

    def __end_search(self) -> None:
        self._search_header_bar.exit()
        note = self._note_list.get_selected_note()
        self._note_list_scroll.get_vadjustment().set_value(0)
        self.__set_search_visible(False)
        self.__on_category_activated(self.__category)
        if note is not None and self.get_root().using_keyboard_navigation:
            self._note_list.select_and_focus_note(note)

    def __update_enabled_actions(self, enable: bool) -> None:
        for action in ("toggle-sidebar", "create-note", "search", "select-notes"):
            action = self.__action_group.lookup(action)
            action.set_property("enabled", enable)

    def __load_older_notes_and_notify(
        self,
        displayed: bool = True,
        time_filtered: bool = True,
        update_section_visibility: bool = False,
    ) -> None:
        """Load the remaining bulk of notes.

        Populate the remaining notes which haven't been shown in the index as they are beyond the
        two month old mark.

        :param bool displayed: The user has requested this load action and the notes should be
            shown in the main index. If false the notes are being loaded for searching and won't
            be displayed in the main index yet (which retains performance in the main list)
        :param bool time_filtered: Whether to initially filter the notes by time (false for
            search)
        :param bool update_section_visibility: Whether section visibility should be updated
        """

        def precache() -> None:
            GLib.idle_add(
                self.__load_older_notes_worker,
                displayed,
                time_filtered,
                update_section_visibility,
            )

        # Translators: Description, notification
        self._notification.show(_("Loading"), immediate=True, callback=precache)

    def __load_older_notes_worker(
        self,
        displayed: bool = True,
        time_filtered: bool = True,
        update_section_visibility: bool = False,
    ) -> None:
        self.__older_notes_model = self.__note_manager.initiate_older_notes_model()
        if time_filtered:
            self.__note_manager.filter_older_notes_by_date()
        loaded_notes = self._note_list.populate_older_notes(
            self.__note_manager.older_notes_model, displayed
        )
        if update_section_visibility:
            self._note_list.refresh_section_visibility(self.__category)

        if loaded_notes:
            if self._stack.get_visible_child() in (self._empty, self._first_start):
                self._stack.set_visible_child(self._note_list_scroll)
        self._notification.hide()

    def __on_initial_load_from_db_complete(
        self, _obj: GObject.Object, older_notes_loaded: bool
    ) -> None:
        self.__invalidate_view()

        # Load all notes at startup if there aren't many in total, or are few in the
        # last two months
        if older_notes_loaded:
            self.__older_notes_model = self.__note_manager.older_notes_model
            self._note_list.populate_older_notes(self.__note_manager.older_notes_model)

        self._stack.set_visible(True)

        self._note_list.refresh_section_visibility(self.__category)

        self._notification.hide()

        self.__initialised = True
        GLib.idle_add(self._note_list.update_category_labels, self.__category)
        if self.__create_note_after_init:
            GLib.idle_add(self.__create_and_open_note)

    def __on_new_note_persisted(self, _obj: GObject.Object) -> None:
        self.__invalidate_view()
        self._note_list.update_category_labels(self.__category)

    def __on_deletion_undo_elapsed(self) -> None:
        self.__note_manager.on_deletion_undo_elapsed(self.__sync_manager.authenticated)

    def __undo_deletion(self) -> None:
        if self.__note_manager.undo_deletion():
            self.__invalidate_view()

    def __get_focused(self) -> Optional[Gtk.Widget]:
        window = self.get_root()
        return window.get_focus()

    def __handle_emptied_category(self, note: Optional[Note] = None) -> bool:
        # If we're not filtered by category this won't be an issue
        if not self.__is_showing_category():
            return False

        # Including "older notes" here as we always show older notes when displaying a category
        view_empty = self.__note_manager.get_filtered_note_count(True) == 0
        if view_empty:
            if note is not None and note.category != "":
                new_category = self._sidebar.select_and_fetch_category(note.category)
            else:
                new_category = self._sidebar.select_and_fetch_category(
                    None, CategorySpecialPurpose.ALL
                )
            self.__on_category_activated(new_category)
            return True

        return False

    def __update_favourites(self, notes: List[Note], value: bool) -> None:
        """Change whether notes are in favourites.

        :param List[Note] notes: Notes to modify
        :param bool value: The new value
        """
        if self.__note_manager.set_and_persist_favourite_for_notes(notes, value):
            self.__sync_manager.sync_now()
            if self.__searching():
                self.__apply_search()
            else:
                self.__invalidate_view()

    def __cancel(self, _action: Gio.SimpleAction, _param: GLib.Variant) -> None:
        """Handle a user cancel action (eg. hitting escape)

        If searching returns to normal view.
        """
        if self._categories_flap.get_reveal_flap():
            self.__close_sidebar()
        elif self._selection_header_bar.active:
            self.__cancel_selection()
        elif self.__searching():
            self.__end_search()

    def __close_sidebar(self) -> None:
        self._categories_flap.set_reveal_flap(False)
        if self.get_root().using_keyboard_navigation:
            self._note_list.grab_focus()

    def __toggle_sidebar(self) -> None:
        if self._categories_flap.get_reveal_flap():
            self.__close_sidebar()
        else:
            self._sidebar.take_focus()
            self._categories_flap.set_reveal_flap(True)

    def __category_style_changed(self, _obj: Gio.Settings, _key: str) -> None:
        self._note_list.update_category_labels(self.__category)

    def __perform_initial_sync(self) -> None:
        self.__initial_sync_delay_complete = True
        self.__sync_manager.sync_now()

    def __delete_notes(self, notes: List[Note]) -> None:
        """Delete the specified notes, flushing any undo-pending notes first.

        :param List[Note] notes: Notes to delete
        """
        self.__sync_manager.flush_pending_deletions()
        self.__note_manager.delete_notes(notes)
        self.update_for_note_deletions(notes)

    def __is_showing_category(self) -> bool:
        return self.__category.special_purpose is None
